document.addEventListener('DOMContentLoaded', async () => {
  const setupDiv = document.getElementById('setup');
  const mainDiv = document.getElementById('main');
  const apiKeyInput = document.getElementById('apiKey');
  const saveKeyBtn = document.getElementById('saveKey');
  const shareBtn = document.getElementById('shareBtn');
  const shareBtnText = document.getElementById('shareBtnText');
  const shareBtnSpinner = document.getElementById('shareBtnSpinner');
  const resultDiv = document.getElementById('result');
  const settingsBtn = document.getElementById('settingsBtn');
  const pageTitleEl = document.getElementById('pageTitle');
  const pageUrlEl = document.getElementById('pageUrl');

  const PROD_SERVER = 'https://seafruit.seafruit-dev.workers.dev';

  // Load saved settings
  const { apiKey } = await chrome.storage.sync.get(['apiKey']);

  if (apiKey) {
    showMainView();
    // Auto-trigger share on open
    triggerShare();
  } else {
    showSetupView();
  }

  async function showMainView() {
    setupDiv.classList.add('hidden');
    mainDiv.classList.remove('hidden');
    resultDiv.classList.add('hidden');

    // Show current tab info
    try {
      const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
      if (tab) {
        pageTitleEl.textContent = tab.title || 'Current Page';
        pageUrlEl.textContent = tab.url || '';
      }
    } catch {
      pageTitleEl.textContent = 'Current Page';
      pageUrlEl.textContent = '';
    }
  }

  function showSetupView() {
    mainDiv.classList.add('hidden');
    setupDiv.classList.remove('hidden');

    // Pre-fill existing values
    if (apiKey) apiKeyInput.value = apiKey;
  }

  // Save settings
  saveKeyBtn.addEventListener('click', async () => {
    const key = apiKeyInput.value.trim();
    if (!key) {
      apiKeyInput.style.borderColor = '#ef4444';
      apiKeyInput.focus();
      return;
    }

    await chrome.storage.sync.set({
      apiKey: key
    });

    showMainView();
    triggerShare();
  });

  // Settings button
  settingsBtn.addEventListener('click', () => {
    showSetupView();
  });

  // Manual Share button logic
  shareBtn.addEventListener('click', triggerShare);

  async function triggerShare() {
    shareBtnText.textContent = 'Sharing...';
    shareBtnSpinner.classList.remove('hidden');
    shareBtn.disabled = true;
    resultDiv.classList.add('hidden');

    try {
      const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });

      // Extract page content
      const [{ result: pageContent }] = await chrome.scripting.executeScript({
        target: { tabId: tab.id },
        func: () => {
          const selection = window.getSelection();
          if (selection && selection.toString().trim()) {
            return selection.toString();
          }
          const main = document.querySelector('main, article, [role="main"], .content, #content');
          const source = main || document.body;
          return source.innerText;
        }
      });

      if (!pageContent || !pageContent.trim()) {
        throw new Error('No content found on this page.');
      }

      const { apiKey: key } = await chrome.storage.sync.get(['apiKey']);
      
      const title = tab.title || 'Untitled';
      const pageUrl = tab.url || '';
      const content = `# ${title}\n\nSource: ${pageUrl}\n\n---\n\n${pageContent}`;

      const response = await fetch(`${PROD_SERVER}/api/share`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${key}`
        },
        body: JSON.stringify({ content })
      });

      if (response.status === 401) {
        throw new Error('Invalid API key. Check your settings.');
      }

      if (response.status === 403) {
        const errorData = await response.json().catch(() => ({}));
        throw new Error(errorData.error || 'Feature locked. Upgrade to Pro.');
      }

      if (!response.ok) {
        throw new Error(`Server error (${response.status})`);
      }

      const data = await response.json();
      
      // Auto-copy to clipboard
      try {
        await navigator.clipboard.writeText(data.url);
        showSuccess(data.url, true);
      } catch (err) {
        console.error('Failed to copy:', err);
        showSuccess(data.url, false);
      }
    } catch (err) {
      showError(err.message);
    } finally {
      shareBtnText.textContent = 'Share Again';
      shareBtnSpinner.classList.add('hidden');
      shareBtn.disabled = false;
    }
  }

  function showSuccess(url, copied) {
    resultDiv.className = 'result-success';
    resultDiv.innerHTML = `
      <div class="result-header">
        <span class="result-label" style="color: var(--success)">${copied ? 'Copied to clipboard!' : 'Shared'}</span>
        <button class="result-copy" id="copyUrl">${copied ? 'Copied!' : 'Copy URL'}</button>
      </div>
      <div class="result-url"><a href="${url}" target="_blank">${url}</a></div>
    `;
    resultDiv.classList.remove('hidden');

    document.getElementById('copyUrl').addEventListener('click', () => {
      navigator.clipboard.writeText(url).then(() => {
        const btn = document.getElementById('copyUrl');
        btn.textContent = 'Copied!';
        btn.classList.add('copied');
        setTimeout(() => {
          btn.textContent = 'Copy URL';
          btn.classList.remove('copied');
        }, 2000);
      });
    });
  }

  function showError(message) {
    resultDiv.className = 'result-error';
    resultDiv.textContent = message;
    resultDiv.classList.remove('hidden');
  }
});
